package integergroup

import (
	"math/big"
	"testing"
)

func cmpValues(calc, expect *big.Int, bits, variable string, t *testing.T) {
	if calc.Cmp(expect) != 0 {
		t.Errorf("Calculated value for %s for %s bit group is not same as expected: %s\n",
			variable, bits, calc.Text(10))
	}
}

func TestM(t *testing.T) {
	m1024 := I1024.ConstM()
	m2048 := I2048.ConstM()
	m3072 := I3072.ConstM()

	expectedM1024 := new(big.Int)
	expectedM1024.SetString(
		"133692765535982424349499845160957985769591080051496769878760125198810878725961537864180017967454437317351593275113191947311781854750725038873659690589675271761307229765099862088336446403690001904873877088799718436019503901547389583394334848296599425103637442262874892284680082984710977844504652305181489393600", 0)
	expectedM2048 := new(big.Int)
	expectedM2048.SetString("11776573954573153074955710691649161959099564861784009358097707182464409468575339293573525192609920506146090859031695650380672658611643346408211033711831166006189784763954768393769222027964502073671061122478597334230912784017038554130494831769380215553278450943693455683703376437808674336725652154214210815382997120646252780682950066782971672297093288452368116263882213413105851850295890710802025421264657320111028747233917008362860704067945126786920565140245810664851612797478373430077125378922098681211716853257742915546777650717348936024091604762771852729812817193964732196386836522074021267205282888458208733372125", 0)
	expectedM3072 := new(big.Int)
	expectedM3072.SetString("3235291780904500472048383187616824728062832132454261592143180577070019309663932974940693737190900308752038759199207026090381932731592352353399736056314911108666648425734031630781913213239854320540639909336327586407281956097687856034139851630074140724840448349381025503625045138868855509149059707326922838943532215004664933389069943646735663811096283199535359562459919428153493554112911510065835609299333943239509420017543033005077185049140680013333339247601201384295810741170961514941022416006100306707401225805481901313368186445835967341045879233767378468558389592908031661732947288708951784727014546987058095217625041620999112277301483674027800875488641892156438867160120932015592782389859151197067280141937787276810415271545076674731551173189014768139305457419105523494466587820357480934263832786780132992298824492210352903915380183288275954306602927551157787002933846495915968261304275024400331956858755625537915033455718", 0)

	cmpValues(m1024.(IntegerElement).e, expectedM1024, "1024", "M", t)
	cmpValues(m2048.(IntegerElement).e, expectedM2048, "2048", "M", t)
	cmpValues(m3072.(IntegerElement).e, expectedM3072, "3072", "M", t)
}

func TestN(t *testing.T) {
	n1024 := I1024.ConstN()
	n2048 := I2048.ConstN()
	n3072 := I3072.ConstN()

	expectedN1024 := new(big.Int)
	expectedN1024.SetString("150024345466173530519975682041188867742370982073129881466303701096866044097476425179151161152054587630855256556972009648705257651636647898694446533641542475968451000872704575361511678956835016931564064989199656781882666271486672654302783948193505426943673841361973991601844052078522112149388388024596736369801", 0)
	expectedN2048 := new(big.Int)
	expectedN2048.SetString("4965258860178670716891104776006270721037652426410686991799935076391990877488535360810232488106686541816630372845709370930562983170226631780261902323195438533088599964341079993228923602599424257975637435711731012162813951400580849895606130198735932937789234450071298351976566569158051951794393797096359012895687565976407388923195237732048329527930178235233704183917002537595387350886984980583416313371923303092880265988461219138991150980160913127996527851471150222399863457258038517161747893825931394843605564211728082611266910041253381284162225936102404112612661791921099243098482469358128474002856306185205658221202", 0)
	expectedN3072 := new(big.Int)
	expectedN3072.SetString("1575147879073131858219703835095259826540910371759353675372378282932746113320353474222262679452713130787535163635555103541360598714274410025703697954899315391974210282375180126663227148165130153449471247506281884014067584801467801513113804524052649745798852448539673992658454700393255602600880424737523053849868326699871886131428237528273026900699978249309833627346564297823579411533804439871997831980060730506404260758565417430511737221603168951042893700101043915319811709512323314665428361576266792946978885524446343237994862586381639436051449243578183554220931803763387943683488528160879838052477981253592252355401396292992200674582025445341225221003969850312591571539418381477642855438920721981212524480686953321602388650837496412530059406707866108472504612616640719828544581175694060463305103140289904814071069713351952468210468583903253721747674516301251875004269504516151682548808982930851446832751402839748361554227121", 0)

	cmpValues(n1024.(IntegerElement).e, expectedN1024, "1024", "N", t)
	cmpValues(n2048.(IntegerElement).e, expectedN2048, "2048", "N", t)
	cmpValues(n3072.(IntegerElement).e, expectedN3072, "3072", "N", t)
}

func TestS(t *testing.T) {
	s1024 := I1024.ConstS()
	s2048 := I2048.ConstS()
	s3072 := I3072.ConstS()

	expectedS1024 := new(big.Int)
	expectedS2048 := new(big.Int)
	expectedS3072 := new(big.Int)

	expectedS1024.SetString("79097103568249206751511928194674286529299310936148863494782470623412886636767249723911543790196162748894233125429155559393554831365931894827483361509891224326020194940099613230307742494697608347687609054068639727042833834601874779609484610294438129798114070101748327949716615952757791844996357332246211803531", 0)
	expectedS2048.SetString("24002301510416124747310925361307841984468431414348754114903162246488618610433721489168356493553860784911914376747273970227784776247591876240145091567361804757971077596290607282962783994304481930381327589987018397373480677182802432469663811205922657326430062482584239098850999982751473303196770823457688111149095084638707398993424236998442210130103343519278720968145482878337387833339759729994022832872986034793626032300916549297881800413421893860565212470513573244757628046092129896219503811665203866310789321409511728375984102314644253194405006353309556672027445678281753779325995588271713838983646641239194317569892", 0)
	expectedS3072.SetString("1566427032507800693534186494933520103872890066346147325498924688203076381361746589602769586892264276947574373423335708922532910880647212955344974516285788173374796575520539145124209707514208027131140665645146594050076085236874384379603810896348043051964465499286812777096805582537066755843783733752726802086215770931603162996191359314246030765889957708336797520260208925476013653014003681145865341296006829522389949227972669666166134506450553648262057148447900776304255530947935394852039129857521828800820740512331690320758327098630085505654269867032453124339454724063084513138652275385213865115875976701889148931327101611858325180345330721433369678170038972854217208013198546133085654848868604487839260321802521302362150714942143503707672923925723732327596290733798426156368336540415896978400837682041151868539251919934919632128865510981259877701332346877862836326512564208855689710874277331645987381438178912709198701091351", 0)

	cmpValues(s1024.(IntegerElement).e, expectedS1024, "1024", "S", t)
	cmpValues(s2048.(IntegerElement).e, expectedS2048, "2048", "S", t)
	cmpValues(s3072.(IntegerElement).e, expectedS3072, "3072", "S", t)

}

func TestMath(t *testing.T) {
	groups := []IntegerGroup{I1024, I2048, I3072}
	for _, group := range groups {
		sb := group.BasePointMult
		e0 := sb(big.NewInt(0)).(IntegerElement)
		if e0.e.Cmp(big.NewInt(1)) != 0 {
			t.Errorf("Any element of group scalarmultiplied by zero should give Identity back")
		}

		e1 := sb(big.NewInt(1)).(IntegerElement)
		e2 := sb(big.NewInt(2)).(IntegerElement)

		a1 := e1.Add(e0)
		if a1.(IntegerElement).e.Cmp(e1.e) != 0 {
			t.Errorf("Element multiplied by Identity should return same element")
		}

		a2 := e1.Add(e1)
		if a2.(IntegerElement).e.Cmp(e2.e) != 0 {
			t.Errorf("Element scalarmultiplied by 2 should be same as element added to itself")
		}

		em1 := sb(new(big.Int).Sub(group.Order(), big.NewInt(1))).(IntegerElement)
		am1 := sb(big.NewInt(-1)).(IntegerElement)
		if em1.e.Cmp(am1.e) != 0 {
			t.Errorf("Scalar multiply by subgroup order-1 is same as multiplying element by -1")
		}

		ez := em1.Add(e1)
		if ez.(IntegerElement).e.Cmp(e0.e) != 0 {
			t.Errorf("Element multiplied by its inverse should result in Identity")
		}

		if e1.ScalarMult(group.Order()).(IntegerElement).e.Cmp(e0.e) != 0 {
			t.Errorf("Multiplying element by subgroup order should result in identity element")
		}

	}

}

func TestToFromBytes(t *testing.T) {
	groups := []IntegerGroup{I1024, I2048, I3072}
	for _, group := range groups {
		base := IntegerElement{params: group.params, e: group.params.g}
		baseBytes := group.ElementToBytes(base)
		e, err := group.ElementFromBytes(baseBytes)
		if err != nil {
			t.Errorf("Failed to reconstruct element from bytes: %s\n", err)
		}

		if e.(IntegerElement).e.Cmp(base.e) != 0 {
			t.Errorf("Reconstructed element is not same as original element\n")
		}

		rs, err := group.RandomScalar()
		if err != nil {
			t.Errorf("Failed to generate random scalar\n")
		}
		er := group.BasePointMult(rs).(IntegerElement)
		erBytes := group.ElementToBytes(er)
		ar, err := group.ElementFromBytes(erBytes)
		if err != nil {
			t.Errorf("Failed to reconstruct element from bytes: %s\n", err)
		}

		if er.e.Cmp(ar.(IntegerElement).e) != 0 {
			t.Errorf("Reconstructed element is not same as original element\n")
		}

		genbytes := make([]byte, len(group.params.p.Bytes()))
		genbytes[0] = 0
		_, err = group.ElementFromBytes(genbytes)
		if err == nil {
			t.Errorf("Should have caused an error trying to generate element from bytes")
		}

		genbytes[0] = 0x02
		_, err = group.ElementFromBytes(genbytes)
		if err == nil {
			t.Errorf("Should have caused an error trying to generate element from bytes")
		}
	}
}

func TestPasswdToScalar(t *testing.T) {
	groups := []IntegerGroup{I1024, I2048, I3072}
	for _, group := range groups {
		pwscalar := group.PasswordToScalar([]byte("password"))
		expected := new(big.Int)
		switch group.params.elementSizeBytes {
		case I1024.params.elementSizeBytes:
			expected.SetString("663780224317275567926254130349768709918789705075", 0)
		case I2048.params.elementSizeBytes:
			expected.SetString("9087812500082240304823557581532115116696050133122505882997099943570", 0)
		case I3072.params.elementSizeBytes:
			expected.SetString("6421227293910067895270565820640017015193799363158329882020568817203002115946", 0)
		}

		if pwscalar.Cmp(expected) != 0 {
			t.Errorf("Calculated password scalar not same as expected for %d bit group\n", group.params.elementSizeBits)
			t.Errorf("Calculated value: %s\n", pwscalar.Text(10))
		}
	}
}
