#' Plot the main DBS contexts
#'
#' @details
#' Plots the number of DBS per main COSMIC context per sample.
#' The contexts are only divided by REF and not by ALT.
#' It takes a tibble with counts as its input. This tibble can be generated by count_dbs_contexts
#' Each sample is plotted in a separate facet.
#' The same y axis can be used for all samples or a separate y axis can be used.
#'
#' @param counts A tibble containing the number of DBS per COSMIC context.
#' @param same_y A boolean describing whether the same y axis should be used for all samples.
#'
#' @return A ggplot figure.
#'
#' @examples
#' ## Get The DBS counts
#' ## See 'count_dbs_contexts()' for more info on how to do this.
#' dbs_counts <- readRDS(system.file("states/blood_dbs_counts.rds",
#'   package = "MutationalPatterns"
#' ))
#'
#' ## Plot contexts
#' plot_main_dbs_contexts(dbs_counts)
#'
#' ## Use the same y axis for all samples.
#' plot_main_dbs_contexts(dbs_counts, same_y = TRUE)
#' @import ggplot2
#' @importFrom magrittr %>%
#' @family DBS
#'
#' @seealso \code{\link{count_dbs_contexts}}, \code{\link{plot_dbs_contexts}}
#'
#' @export
plot_main_dbs_contexts <- function(counts, same_y = FALSE) {

  # These variables use non standard evaluation.
  # To avoid R CMD check complaints we initialize them to NULL.
  count <- REF <- ALT <- muttype_total <- sample <- NULL

  # Transform to data frame
  counts <- counts %>%
    as.data.frame() %>%
    tibble::rownames_to_column("muttype_total") %>%
    tidyr::separate(muttype_total, c("REF", "ALT"), sep = "_") %>%
    dplyr::mutate(REF = factor(REF, levels = BiocGenerics::unique(REF)))

  # Summarize by REF and turn into long format.
  counts_main <- counts %>%
    dplyr::select(-ALT) %>%
    dplyr::group_by(REF) %>%
    dplyr::summarise_all(list(~ sum(.))) %>%
    tidyr::gather(key = "sample", value = "count", -REF) %>% 
    dplyr::mutate(sample = factor(sample, levels = unique(sample)))

  # Count nr muts per sample for facet
  nr_muts <- counts_main %>%
    dplyr::group_by(sample) %>%
    dplyr::summarise(nr_muts = sum(count))
  facet_labs_y <- stringr::str_c(nr_muts$sample, " (n = ", nr_muts$nr_muts, ")")
  names(facet_labs_y) <- nr_muts$sample

  # Plot settings
  if (same_y) {
    facet_scale <- "free_x"
  } else {
    facet_scale <- "free"
  }

  # Create plot
  fig <- ggplot(counts_main, aes(x = REF, y = count, fill = REF)) +
    geom_bar(stat = "identity") +
    facet_grid(sample ~ ., labeller = labeller(sample = facet_labs_y), scales = facet_scale) +
    labs(x = "", y = "Nr of DBSs") +
    scale_fill_manual(guide = "none", values = DBS_COLORS) +
    theme_bw() +
    theme(
      panel.grid.major.x = element_blank(),
      panel.grid.minor.y = element_blank(),
      axis.text.x = element_text(angle = 90, hjust = 1, vjust = 0.5)
    )
  return(fig)
}
