// SPDX-License-Identifier: GPL-2.0+
/*
 * MediaTek BootROM NAND header definitions
 *
 * Copyright (C) 2022 MediaTek Inc.
 * Author: Weijie Gao <weijie.gao@mediatek.com>
 */

#include <stdint.h>
#include <string.h>
#include "imagetool.h"
#include "mtk_image.h"
#include "mtk_nand_headers.h"

/* NAND header for SPI-NAND with 2KB page + 64B spare */
static const union nand_boot_header snand_hdr_2k_64_data = {
	.data = {
		0x42, 0x4F, 0x4F, 0x54, 0x4C, 0x4F, 0x41, 0x44,
		0x45, 0x52, 0x21, 0x00, 0x56, 0x30, 0x30, 0x36,
		0x4E, 0x46, 0x49, 0x49, 0x4E, 0x46, 0x4F, 0x00,
		0x00, 0x00, 0x00, 0x08, 0x03, 0x00, 0x40, 0x00,
		0x40, 0x00, 0x00, 0x08, 0x10, 0x00, 0x16, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x7B, 0xC4, 0x17, 0x9D,
		0xCA, 0x42, 0x90, 0xD0, 0x98, 0xD0, 0xE0, 0xF7,
		0xDB, 0xCD, 0x16, 0xF6, 0x03, 0x73, 0xD2, 0xB8,
		0x93, 0xB2, 0x56, 0x5A, 0x84, 0x6E, 0x00, 0x00
	}
};

/* NAND header for SPI-NAND with 2KB page + 120B/128B spare */
static const union nand_boot_header snand_hdr_2k_128_data = {
	.data = {
		0x42, 0x4F, 0x4F, 0x54, 0x4C, 0x4F, 0x41, 0x44,
		0x45, 0x52, 0x21, 0x00, 0x56, 0x30, 0x30, 0x36,
		0x4E, 0x46, 0x49, 0x49, 0x4E, 0x46, 0x4F, 0x00,
		0x00, 0x00, 0x00, 0x08, 0x05, 0x00, 0x70, 0x00,
		0x40, 0x00, 0x00, 0x08, 0x10, 0x00, 0x16, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x90, 0x28, 0xED, 0x13,
		0x7F, 0x12, 0x22, 0xCD, 0x3D, 0x06, 0xF1, 0xB3,
		0x6F, 0x2E, 0xD9, 0xA0, 0x9D, 0x7A, 0xBD, 0xD7,
		0xB3, 0x28, 0x3C, 0x13, 0xDB, 0x4E, 0x00, 0x00
	}
};

/* NAND header for SPI-NAND with 4KB page + 256B spare */
static const union nand_boot_header snand_hdr_4k_256_data = {
	.data = {
		0x42, 0x4F, 0x4F, 0x54, 0x4C, 0x4F, 0x41, 0x44,
		0x45, 0x52, 0x21, 0x00, 0x56, 0x30, 0x30, 0x36,
		0x4E, 0x46, 0x49, 0x49, 0x4E, 0x46, 0x4F, 0x00,
		0x00, 0x00, 0x00, 0x10, 0x05, 0x00, 0xE0, 0x00,
		0x40, 0x00, 0x00, 0x08, 0x10, 0x00, 0x16, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x47, 0xED, 0x0E, 0xC3,
		0x83, 0xBF, 0x41, 0xD2, 0x85, 0x21, 0x97, 0x57,
		0xC4, 0x2E, 0x6B, 0x7A, 0x40, 0xE0, 0xCF, 0x8F,
		0x37, 0xBD, 0x17, 0xB6, 0xC7, 0xFE, 0x00, 0x00
	}
};

/* NAND header for Parallel NAND 1Gb with 2KB page + 64B spare */
static const union nand_boot_header nand_hdr_1gb_2k_64_data = {
	.data = {
		0x42, 0x4F, 0x4F, 0x54, 0x4C, 0x4F, 0x41, 0x44,
		0x45, 0x52, 0x21, 0x00, 0x56, 0x30, 0x30, 0x36,
		0x4E, 0x46, 0x49, 0x49, 0x4E, 0x46, 0x4F, 0x00,
		0x00, 0x00, 0x00, 0x08, 0x05, 0x00, 0x40, 0x00,
		0x40, 0x00, 0x00, 0x04, 0x0B, 0x00, 0x11, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x12, 0x28, 0x1C, 0x12,
		0x8F, 0xFD, 0xF8, 0x32, 0x6F, 0x6D, 0xCF, 0x6C,
		0xDA, 0x21, 0x70, 0x8C, 0xDA, 0x0A, 0x22, 0x82,
		0xAA, 0x59, 0xFA, 0x7C, 0x42, 0x2D, 0x00, 0x00
	}
};

/* NAND header for Parallel NAND 2Gb with 2KB page + 64B spare */
static const union nand_boot_header nand_hdr_2gb_2k_64_data = {
	.data = {
		0x42, 0x4F, 0x4F, 0x54, 0x4C, 0x4F, 0x41, 0x44,
		0x45, 0x52, 0x21, 0x00, 0x56, 0x30, 0x30, 0x36,
		0x4E, 0x46, 0x49, 0x49, 0x4E, 0x46, 0x4F, 0x00,
		0x00, 0x00, 0x00, 0x08, 0x05, 0x00, 0x40, 0x00,
		0x40, 0x00, 0x00, 0x08, 0x0B, 0x00, 0x11, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x20, 0x9C, 0x3D, 0x2D,
		0x7B, 0x68, 0x63, 0x52, 0x2E, 0x04, 0x63, 0xF1,
		0x35, 0x4E, 0x44, 0x3E, 0xF8, 0xAC, 0x9B, 0x95,
		0xAB, 0xFE, 0xE4, 0xE1, 0xD5, 0xF9, 0x00, 0x00
	}
};

/* NAND header for Parallel NAND 4Gb with 2KB page + 64B spare */
static const union nand_boot_header nand_hdr_4gb_2k_64_data = {
	.data = {
		0x42, 0x4F, 0x4F, 0x54, 0x4C, 0x4F, 0x41, 0x44,
		0x45, 0x52, 0x21, 0x00, 0x56, 0x30, 0x30, 0x36,
		0x4E, 0x46, 0x49, 0x49, 0x4E, 0x46, 0x4F, 0x00,
		0x00, 0x00, 0x00, 0x08, 0x05, 0x00, 0x40, 0x00,
		0x40, 0x00, 0x00, 0x10, 0x0B, 0x00, 0x11, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xE3, 0x0F, 0x86, 0x32,
		0x68, 0x05, 0xD9, 0xC8, 0x13, 0xDF, 0xC5, 0x0B,
		0x35, 0x3A, 0x68, 0xA5, 0x3C, 0x0C, 0x73, 0x87,
		0x63, 0xB0, 0xBE, 0xCC, 0x84, 0x47, 0x00, 0x00
	}
};

/* NAND header for Parallel NAND 2Gb with 2KB page + 128B spare */
static const union nand_boot_header nand_hdr_2gb_2k_128_data = {
	.data = {
		0x42, 0x4F, 0x4F, 0x54, 0x4C, 0x4F, 0x41, 0x44,
		0x45, 0x52, 0x21, 0x00, 0x56, 0x30, 0x30, 0x36,
		0x4E, 0x46, 0x49, 0x49, 0x4E, 0x46, 0x4F, 0x00,
		0x00, 0x00, 0x00, 0x08, 0x05, 0x00, 0x70, 0x00,
		0x40, 0x00, 0x00, 0x08, 0x0B, 0x00, 0x11, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x01, 0xA5, 0xE9, 0x5A,
		0xDF, 0x58, 0x62, 0x41, 0xD6, 0x26, 0x77, 0xBC,
		0x76, 0x1F, 0x27, 0x4E, 0x4F, 0x6C, 0xC3, 0xF0,
		0x36, 0xDE, 0xD9, 0xB3, 0xFF, 0x93, 0x00, 0x00
	}
};

/* NAND header for Parallel NAND 4Gb with 2KB page + 128B spare */
static const union nand_boot_header nand_hdr_4gb_2k_128_data = {
	.data = {
		0x42, 0x4F, 0x4F, 0x54, 0x4C, 0x4F, 0x41, 0x44,
		0x45, 0x52, 0x21, 0x00, 0x56, 0x30, 0x30, 0x36,
		0x4E, 0x46, 0x49, 0x49, 0x4E, 0x46, 0x4F, 0x00,
		0x00, 0x00, 0x00, 0x08, 0x05, 0x00, 0x70, 0x00,
		0x40, 0x00, 0x00, 0x10, 0x0B, 0x00, 0x11, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xC2, 0x36, 0x52, 0x45,
		0xCC, 0x35, 0xD8, 0xDB, 0xEB, 0xFD, 0xD1, 0x46,
		0x76, 0x6B, 0x0B, 0xD5, 0x8B, 0xCC, 0x2B, 0xE2,
		0xFE, 0x90, 0x83, 0x9E, 0xAE, 0x2D, 0x00, 0x00
	}
};

/* HSM BROM NAND header for SPI NAND with 2KB page + 64B spare */
static const union hsm_nand_boot_header hsm_nand_hdr_2k_64_data = {
	.data = {
		0x4E, 0x41, 0x4E, 0x44, 0x43, 0x46, 0x47, 0x21,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x04, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x08, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
		0x40, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
		0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
		0xFF, 0x00, 0x00, 0x00, 0x21, 0xD2, 0xEE, 0xF6,
		0xAE, 0xDD, 0x5E, 0xC2, 0x82, 0x8E, 0x9A, 0x62,
		0x09, 0x8E, 0x80, 0xE2, 0x37, 0x0D, 0xC9, 0xFA,
		0xA9, 0xDD, 0xFC, 0x92, 0x34, 0x2A, 0xED, 0x51,
		0xA4, 0x1B, 0xF7, 0x63, 0xCC, 0x5A, 0xC7, 0xFB,
		0xED, 0x21, 0x02, 0x23, 0x51, 0x31
	}
};

/* HSM BROM NAND header for SPI NAND with 2KB page + 128B spare */
static const union hsm_nand_boot_header hsm_nand_hdr_2k_128_data = {
	.data = {
		0x4E, 0x41, 0x4E, 0x44, 0x43, 0x46, 0x47, 0x21,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x04, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x08, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
		0x40, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
		0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
		0xFF, 0x00, 0x00, 0x00, 0x71, 0x7f, 0x71, 0xAC,
		0x42, 0xD0, 0x5B, 0xD2, 0x12, 0x81, 0x15, 0x0A,
		0x0C, 0xD4, 0xF6, 0x32, 0x1E, 0x63, 0xE7, 0x81,
		0x8A, 0x7F, 0xDE, 0xF9, 0x4B, 0x91, 0xEC, 0xC2,
		0x70, 0x00, 0x7F, 0x57, 0xAF, 0xDC, 0xE4, 0x24,
		0x57, 0x09, 0xBC, 0xC5, 0x35, 0xDC
	}
};

/* HSM BROM NAND header for SPI NAND with 4KB page + 256B spare */
static const union hsm_nand_boot_header hsm_nand_hdr_4k_256_data = {
	.data = {
		0x4E, 0x41, 0x4E, 0x44, 0x43, 0x46, 0x47, 0x21,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x04, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
		0x40, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
		0x0C, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
		0xFF, 0x00, 0x00, 0x00, 0x62, 0x04, 0xD6, 0x1F,
		0x2B, 0x57, 0x7A, 0x2D, 0xFE, 0xBB, 0x4A, 0x50,
		0xEC, 0xF8, 0x70, 0x1A, 0x44, 0x15, 0xF6, 0xA2,
		0x8E, 0xB0, 0xFD, 0xFA, 0xDC, 0xAA, 0x5A, 0x4E,
		0xCB, 0x8E, 0xC9, 0x72, 0x08, 0xDC, 0x20, 0xB9,
		0x98, 0xC8, 0x82, 0xD8, 0xBE, 0x44
	}
};

/* HSM2.0 BROM NAND header for SPI NAND with 2KB page + 64B spare */
static const union hsm20_nand_boot_header hsm20_nand_hdr_2k_64_data = {
	.data = {
		0x4E, 0x41, 0x4E, 0x44, 0x43, 0x46, 0x47, 0x21,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00,
		0x00, 0x04, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x08, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
		0x40, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
		0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
		0x01, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x5F, 0x4B, 0xB2, 0x5B, 0x8B, 0x1C, 0x35, 0xDA,
		0x83, 0xE6, 0x6C, 0xC3, 0xFB, 0x8C, 0x78, 0x23,
		0xD0, 0x89, 0x24, 0xD9, 0x6C, 0x35, 0x2C, 0x5D,
		0x8F, 0xBB, 0xFC, 0x10, 0xD0, 0xE2, 0x22, 0x7D,
		0xC8, 0x97, 0x9A, 0xEF, 0xC6, 0xB5, 0xA7, 0x4E,
		0x4E, 0x0E
	}
};

/* HSM2.0 BROM NAND header for SPI NAND with 2KB page + 128B spare */
static const union hsm20_nand_boot_header hsm20_nand_hdr_2k_128_data = {
	.data = {
		0x4E, 0x41, 0x4E, 0x44, 0x43, 0x46, 0x47, 0x21,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00,
		0x00, 0x04, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x08, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
		0x40, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
		0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
		0x01, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0xF8, 0x7E, 0xC1, 0x5D, 0x61, 0x54, 0xEA, 0x9F,
		0x5E, 0x66, 0x39, 0x66, 0x21, 0xFF, 0x8C, 0x3B,
		0xBE, 0xA7, 0x5A, 0x9E, 0xD7, 0xBD, 0x9E, 0x89,
		0xEE, 0x7E, 0x10, 0x31, 0x9A, 0x1D, 0x82, 0x49,
		0xA3, 0x4E, 0xD8, 0x47, 0xD7, 0x19, 0xF4, 0x2D,
		0x8E, 0x53
	}
};

/* HSM2.0 BROM NAND header for SPI NAND with 4KB page + 256B spare */
static const union hsm20_nand_boot_header hsm20_nand_hdr_4k_256_data = {
	.data = {
		0x4E, 0x41, 0x4E, 0x44, 0x43, 0x46, 0x47, 0x21,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00,
		0x00, 0x04, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
		0x40, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
		0x0C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
		0x01, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00,
		0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x79, 0x01, 0x1F, 0x86, 0x62, 0x6A, 0x43, 0xAE,
		0xE6, 0xF8, 0xDD, 0x5B, 0x29, 0xB7, 0xA2, 0x7F,
		0x29, 0x72, 0x54, 0x37, 0xBE, 0x50, 0xD4, 0x24,
		0xAB, 0x60, 0xF4, 0x44, 0x97, 0x3B, 0x65, 0x21,
		0x73, 0x24, 0x1F, 0x93, 0x0E, 0x9E, 0x96, 0x88,
		0x78, 0x6C
	}
};

/* SPIM-NAND header for SPI NAND with 2KB page + 64B spare */
static const union spim_nand_boot_header spim_nand_hdr_2k_64_data = {
	.data = {
		0x53, 0x50, 0x49, 0x4e, 0x41, 0x4e, 0x44, 0x21,
		0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
		0x00, 0x08, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
		0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x20, 0x30,
		0x01, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	}
};

/* SPIM-NAND header for SPI NAND with 2KB page + 128B spare */
static const union spim_nand_boot_header spim_nand_hdr_2k_128_data = {
	.data = {
		0x53, 0x50, 0x49, 0x4e, 0x41, 0x4e, 0x44, 0x21,
		0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
		0x00, 0x08, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
		0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x20, 0x30,
		0x01, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	}
};

/* SPIM-NAND header for SPI NAND with 4KB page + 256B spare */
static const union spim_nand_boot_header spim_nand_hdr_4k_256_data = {
	.data = {
		0x53, 0x50, 0x49, 0x4e, 0x41, 0x4e, 0x44, 0x21,
		0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
		0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
		0x40, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x20, 0x30,
		0x01, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	}
};

struct nand_header_type {
	const char *name;
	enum nand_boot_header_type type;
	union {
		const union nand_boot_header *ap;
		const union hsm_nand_boot_header *hsm;
		const union hsm20_nand_boot_header *hsm20;
		const union spim_nand_boot_header *spim;
	};
} nand_headers[] = {
	{
		.name = "2k+64",
		.type = NAND_BOOT_AP_HEADER,
		.ap = &snand_hdr_2k_64_data,
	}, {
		.name = "2k+120",
		.type = NAND_BOOT_AP_HEADER,
		.ap = &snand_hdr_2k_128_data,
	}, {
		.name = "2k+128",
		.type = NAND_BOOT_AP_HEADER,
		.ap = &snand_hdr_2k_128_data,
	}, {
		.name = "4k+256",
		.type = NAND_BOOT_AP_HEADER,
		.ap = &snand_hdr_4k_256_data,
	}, {
		.name = "1g:2k+64",
		.type = NAND_BOOT_AP_HEADER,
		.ap = &nand_hdr_1gb_2k_64_data,
	}, {
		.name = "2g:2k+64",
		.type = NAND_BOOT_AP_HEADER,
		.ap = &nand_hdr_2gb_2k_64_data,
	}, {
		.name = "4g:2k+64",
		.type = NAND_BOOT_AP_HEADER,
		.ap = &nand_hdr_4gb_2k_64_data,
	}, {
		.name = "2g:2k+128",
		.type = NAND_BOOT_AP_HEADER,
		.ap = &nand_hdr_2gb_2k_128_data,
	}, {
		.name = "4g:2k+128",
		.type = NAND_BOOT_AP_HEADER,
		.ap = &nand_hdr_4gb_2k_128_data,
	}, {
		.name = "hsm:2k+64",
		.type = NAND_BOOT_HSM_HEADER,
		.hsm = &hsm_nand_hdr_2k_64_data,
	}, {
		.name = "hsm:2k+128",
		.type = NAND_BOOT_HSM_HEADER,
		.hsm = &hsm_nand_hdr_2k_128_data,
	}, {
		.name = "hsm:4k+256",
		.type = NAND_BOOT_HSM_HEADER,
		.hsm = &hsm_nand_hdr_4k_256_data,
	},  {
		.name = "hsm20:2k+64",
		.type = NAND_BOOT_HSM20_HEADER,
		.hsm20 = &hsm20_nand_hdr_2k_64_data,
	}, {
		.name = "hsm20:2k+128",
		.type = NAND_BOOT_HSM20_HEADER,
		.hsm20 = &hsm20_nand_hdr_2k_128_data,
	}, {
		.name = "hsm20:4k+256",
		.type = NAND_BOOT_HSM20_HEADER,
		.hsm20 = &hsm20_nand_hdr_4k_256_data,
	}, {
		.name = "spim:2k+64",
		.type = NAND_BOOT_SPIM_HEADER,
		.spim = &spim_nand_hdr_2k_64_data,
	}, {
		.name = "spim:2k+128",
		.type = NAND_BOOT_SPIM_HEADER,
		.spim = &spim_nand_hdr_2k_128_data,
	}, {
		.name = "spim:4k+256",
		.type = NAND_BOOT_SPIM_HEADER,
		.spim = &spim_nand_hdr_4k_256_data,
	}
};

const struct nand_header_type *mtk_nand_header_find(const char *name)
{
	uint32_t i;

	for (i = 0; i < ARRAY_SIZE(nand_headers); i++) {
		if (!strcmp(nand_headers[i].name, name))
			return &nand_headers[i];
	}

	return NULL;
}

uint32_t mtk_nand_header_size(const struct nand_header_type *hdr_nand)
{
	switch (hdr_nand->type) {
	case NAND_BOOT_HSM_HEADER:
		return le32_to_cpu(hdr_nand->hsm->page_size);

	case NAND_BOOT_HSM20_HEADER:
		return le32_to_cpu(hdr_nand->hsm20->page_size);

	case NAND_BOOT_SPIM_HEADER:
		return le32_to_cpu(hdr_nand->spim->page_size);

	default:
		return 2 * le16_to_cpu(hdr_nand->ap->pagesize);
	}
}

static int mtk_nand_header_ap_info(const void *ptr,
				   struct nand_header_info *info)
{
	union nand_boot_header *nh = (union nand_boot_header *)ptr;

	if (strncmp(nh->version, NAND_BOOT_VERSION, sizeof(nh->version)) ||
	    strcmp(nh->id, NAND_BOOT_ID))
		return -1;

	info->page_size = le16_to_cpu(nh->pagesize);
	info->spare_size = le16_to_cpu(nh->oobsize);
	info->gfh_offset = 2 * info->page_size;
	info->snfi = true;

	return 0;
}

static int mtk_nand_header_hsm_info(const void *ptr,
				    struct nand_header_info *info)
{
	union hsm_nand_boot_header *nh = (union hsm_nand_boot_header *)ptr;

	info->page_size = le16_to_cpu(nh->page_size);
	info->spare_size = le16_to_cpu(nh->spare_size);
	info->gfh_offset = info->page_size;
	info->snfi = true;

	return 1;
}

static int mtk_nand_header_spim_info(const void *ptr,
				     struct nand_header_info *info)
{
	union spim_nand_boot_header *nh = (union spim_nand_boot_header *)ptr;

	info->page_size = le16_to_cpu(nh->page_size);
	info->spare_size = le16_to_cpu(nh->spare_size);
	info->gfh_offset = info->page_size;
	info->snfi = false;

	return 1;
}

int mtk_nand_header_info(const void *ptr, struct nand_header_info *info)
{
	if (!strcmp((char *)ptr, NAND_BOOT_NAME))
		return mtk_nand_header_ap_info(ptr, info);
	else if (!strncmp((char *)ptr, HSM_NAND_BOOT_NAME, 8))
		return mtk_nand_header_hsm_info(ptr, info);
	else if (!strncmp((char *)ptr, SPIM_NAND_BOOT_NAME, 8))
		return mtk_nand_header_spim_info(ptr, info);

	return -1;
}

bool is_mtk_nand_header(const void *ptr)
{
	struct nand_header_info info;

	if (mtk_nand_header_info(ptr, &info) >= 0)
		return true;

	return false;
}

static uint16_t crc16(const uint8_t *p, uint32_t len)
{
	uint16_t crc = 0x4f4e;
	uint32_t i;

	while (len--) {
		crc ^= *p++ << 8;
		for (i = 0; i < 8; i++)
			crc = (crc << 1) ^ ((crc & 0x8000) ? 0x8005 : 0);
	}

	return crc;
}

static uint32_t mtk_nand_header_put_ap(const struct nand_header_type *hdr_nand,
				       void *ptr)
{
	int i;

	/* NAND device header, repeat 4 times */
	for (i = 0; i < 4; i++) {
		memcpy(ptr, hdr_nand->ap, sizeof(*hdr_nand->ap));
		ptr += sizeof(*hdr_nand->ap);
	}

	return le16_to_cpu(hdr_nand->ap->pagesize);
}

static uint32_t mtk_nand_header_put_hsm(const struct nand_header_type *hdr_nand,
					void *ptr)
{
	memcpy(ptr, hdr_nand->hsm, sizeof(*hdr_nand->hsm));
	return 0;
}

static uint32_t mtk_nand_header_put_hsm20(const struct nand_header_type *hdr_nand,
					  void *ptr)
{
	memcpy(ptr, hdr_nand->hsm20, sizeof(*hdr_nand->hsm20));
	return 0;
}

static uint32_t mtk_nand_header_put_spim(const struct nand_header_type *hdr_nand,
					 void *ptr)
{
	uint16_t crc;

	memcpy(ptr, hdr_nand->spim, sizeof(*hdr_nand->spim));

	crc = crc16(ptr, 0x4e);
	memcpy(ptr + 0x4e, &crc, sizeof(uint16_t));

	return 0;
}

uint32_t mtk_nand_header_put(const struct nand_header_type *hdr_nand, void *ptr)
{
	switch (hdr_nand->type) {
	case NAND_BOOT_HSM_HEADER:
		return mtk_nand_header_put_hsm(hdr_nand, ptr);

	case NAND_BOOT_HSM20_HEADER:
		return mtk_nand_header_put_hsm20(hdr_nand, ptr);

	case NAND_BOOT_SPIM_HEADER:
		return mtk_nand_header_put_spim(hdr_nand, ptr);

	default:
		return mtk_nand_header_put_ap(hdr_nand, ptr);
	}
}
